#!/bin/bash
# Ensure the script stops if a command fails
set -e

# --- File Selection ---

# Use AppleScript to prompt the user to select the document file.
# We check the exit code to see if the user clicked "Cancel".
doc_file_path=$(osascript -e '
try
    POSIX path of (choose file of type {"public.rtf", "public.plain-text"} with prompt "Please select the document file (RTF or TXT):")
end try
')

# If doc_file_path is empty, the user cancelled.
if [[ -z "$doc_file_path" ]]; then
    echo "No document file selected. Exiting."
    exit 1
fi

# Use AppleScript to prompt the user to select one or more audio files.
# "public.audio" allows for selection of MP3, M4A, WAV, etc.
audio_files_str=$(osascript -e '
try
    set audio_files to choose file of type {"public.audio"} with prompt "Please select one or more audio files:" with multiple selections allowed
    set audio_paths to ""
    repeat with a_file in audio_files
        set audio_paths to audio_paths & POSIX path of a_file & "\n"
    end repeat
    return audio_paths
end try
')

# If audio_files_str is empty, the user cancelled.
if [[ -z "$audio_files_str" ]]; then
    echo "No audio files selected. Exiting."
    exit 1
fi

# --- Zipping Process ---

# Read the newline-separated string of paths into an array.
# This is the compatible way to do it, avoiding the 'mapfile' command.
audio_files=()
while IFS= read -r line; do
    # Add the line to the array, but only if it's not an empty line
    if [[ -n "$line" ]]; then
        audio_files+=("$line")
    fi
done <<< "$audio_files_str"


# Loop through each selected audio file.
for audio_file in "${audio_files[@]}"; do
    # Ensure the file path is valid.
    if [[ ! -e "$audio_file" ]]; then
        echo "Audio file not found: $audio_file"
        continue
    fi

    # Get the directory and the base name of the audio file (without extension).
    dir_name=$(dirname "$audio_file")
    base_name=$(basename "${audio_file%.*}")
    
    # Define the output zip file name, placing it in the same directory as the audio file.
    zip_name="${dir_name}/${base_name}.zip"
    
    # Create the zip archive. Use "-j" to junk the paths, storing only the files.
    # This prevents the zip file from containing the full directory structure.
    zip -j "$zip_name" "$audio_file" "$doc_file_path"
    
    echo "Created archive: $zip_name"
done

echo "Process complete."